#!/bin/sh
# SPDX-License-Identifier: BSD-2-Clause
# SPDX-FileCopyrightText: © 2025 Antoine Le Gonidec <vv221@dotslashplay.it>
set -o errexit

###
# Total Annihilation
# send your bug reports to bugs@dotslashplay.it
###

script_version=20251026.4

PLAYIT_COMPATIBILITY_LEVEL='2.33'

GAME_ID='total-annihilation'
GAME_NAME='Total Annihilation'

ARCHIVE_BASE_0_NAME='setup_total_annihilation_commander_pack_3.1_(22139).exe'
ARCHIVE_BASE_0_MD5='b89b9f867239d7e7524bfb920d52b921'
ARCHIVE_BASE_0_TYPE='innosetup'
ARCHIVE_BASE_0_SIZE='1082257'
ARCHIVE_BASE_0_VERSION='3.1-gog22139'
ARCHIVE_BASE_0_URL='https://www.gog.com/game/total_anihilation_commander_pack'

CONTENT_PATH_DEFAULT='.'
## totala.exe requirements
CONTENT_LIBS_BIN_FILES='
smackw32.DLL
WIN32.dll'
## WIN32.dll requirements
CONTENT_LIBS0_BIN_FILES='
audiere.dll'
CONTENT_GAME_BIN_FILES='
tae.exe
totala.exe'
CONTENT_GAME_DATA_FILES='
bitmaps
camps
data
music
rev31.gp3
example.tdf
*.ccx
*.hpi
*.ufo'
CONTENT_DOC_DATA_FILES='
btreadme.doc
btreadme.txt
readme.doc
readme.txt
taccread.doc
taccread.txt
taeread.doc
taeread.txt
tae.hlp
manual.pdf'
CONTENT_DOC0_DATA_PATH='docs'
CONTENT_DOC0_DATA_FILES='
afark.txt
aflea.txt
ascarab.txt
cnecro.txt
cormabm.txt
corplas.txt'

USER_PERSISTENT_DIRECTORIES='
savegame'
## The game engine fails to play music if it has no write access to it.
USER_PERSISTENT_DIRECTORIES="$USER_PERSISTENT_DIRECTORIES
music"

## Set game text language.
WINE_REGISTRY_INIT_L10N_FR='
[HKEY_CURRENT_USER\Software\Cavedog Entertainment\Total Annihilation]
"Language"="french"'
WINE_REGISTRY_INIT_L10N_EN='
[HKEY_CURRENT_USER\Software\Cavedog Entertainment\Total Annihilation]
"Language"="english"'

APP_MAIN_EXE='totala.exe'

APP_EDITOR_ID="${GAME_ID}-editor"
APP_EDITOR_NAME="$GAME_NAME - Map and Mission Editor"
APP_EDITOR_EXE='totala.exe'

PACKAGES_LIST='
PKG_BIN
PKG_L10N_FR
PKG_L10N_EN
PKG_DATA'

PKG_BIN_ARCH='32'
PKG_BIN_DEPENDENCIES_SIBLINGS='
PKG_L10N
PKG_DATA'

PKG_L10N_ID="${GAME_ID}-l10n"
PKG_L10N_FR_ID="${PKG_L10N_ID}-fr"
PKG_L10N_EN_ID="${PKG_L10N_ID}-en"
PKG_L10N_PROVIDES="
$PKG_L10N_ID"
PKG_L10N_DESCRIPTION_FR='French localisation'
PKG_L10N_DESCRIPTION_EN='English localisation'

PKG_DATA_ID="${GAME_ID}-data"
PKG_DATA_DESCRIPTION='data'

# Load common functions.

PLAYIT_LIB_PATHS="
$PWD
${XDG_DATA_HOME:="${HOME}/.local/share"}/play.it
/usr/local/share/games/play.it
/usr/local/share/play.it
/usr/share/games/play.it
/usr/share/play.it"

if [ -z "$PLAYIT_LIB2" ]; then
	for playit_lib_path in $PLAYIT_LIB_PATHS; do
		if [ -e "${playit_lib_path}/libplayit2.sh" ]; then
			PLAYIT_LIB2="${playit_lib_path}/libplayit2.sh"
			break
		fi
	done
fi
if [ -z "$PLAYIT_LIB2" ]; then
	printf '\n\033[1;31mError:\033[0m\n'
	printf 'libplayit2.sh not found.\n'
	exit 1
fi
# shellcheck source=libplayit2.sh
. "$PLAYIT_LIB2"

# Run default initialisation actions.

initialization_default "$@"

# Extract game data.

archive_extraction_default

# Include game data.

content_inclusion_icons 'PKG_DATA'
content_inclusion_default

# Set game text language through Windows registry scripts.

for package in 'PKG_L10N_FR' 'PKG_L10N_EN'; do
	## TODO: Compatibility level 2.34 adds context support to WINE_REGISTRY_INIT.
	WINE_REGISTRY_INIT=$(
		set_current_package "$package"
		context_value 'WINE_REGISTRY_INIT'
	)
	# shellcheck disable=SC2218
	wine_registry_script_write "$package"
done
## Prevent the registry script from being written again in PKG_BIN.
wine_registry_script_write() { return 0; }

# Write launchers.

## Install shipped libraries.
wineprefix_init_custom() {
	cat <<- EOF
	# Install shipped libraries.
	ln --force --symbolic $(path_libraries)/* "\${PATH_PREFIX}"
	EOF
}

launchers_generation

# Build packages.

packages_generation
case "$(messages_language)" in
	('fr')
		lang_string='version %s :'
		lang_fr='française'
		lang_en='anglaise'
	;;
	('en'|*)
		lang_string='%s version:'
		lang_fr='French'
		lang_en='English'
	;;
esac
printf '\n'
printf "$lang_string" "$lang_fr"
print_instructions 'PKG_BIN' 'PKG_DATA' 'PKG_L10N_FR'
printf "$lang_string" "$lang_en"
print_instructions 'PKG_BIN' 'PKG_DATA' 'PKG_L10N_EN'

# Clean up.

working_directory_cleanup

exit 0
